/*
 * Routines for running in standalone mode
 */
#include <stdio.h>
#include <unistd.h>
#include <netinet/in.h>

#include "libfma.h"
#include "lf_scheduler.h"
#include "lf_fabric.h"
#include "lf_topo_map.h"
#include "lf_fma_flags.h"
#include "libmyri.h"

#include "fma.h"
#include "fma_fabric.h"
#include "fma_myri.h"
#include "fma_map.h"
#include "fma_fms.h"
#include "fma_standalone.h"
#include "fma_tunnel.h"
#include "fma_verify.h"
#include "fma_proxy.h"

/*
 * Local prototypes
 */
static void fma_standalone_map_timeout(void *v);
static void fma_standalone_map_request_timeout(void *v);
static void fma_request_remap_from_mapper(char *why);
#if 0
static int fma_is_link_xbar_xbar(int topo_index);
#endif
static int fma_compare_levels( int level1, lf_mac_addr_t mac_addr1,
    int level2, lf_mac_addr_t mac_addr2);
static void fma_mapping_grace_expired(void *v);
static void fma_standalone_map_distribution_complete(void);
static void fma_local_map_distribution_complete(void);
static void fma_standalone_dist_timeout(void *v);
static void fma_report_dist_results(void);

/*
 * Initialize static things that depend on nothing else
 */
void
fma_init_standalone_vars()
{
  struct fma_standalone_data *sdp;

  LF_CALLOC(A.stand, struct fma_standalone_data, 1);
  sdp = A.stand;

  return;

 except:
  fma_perror_exit(1);
}

void
fma_init_standalone()
{
}

void
fma_enter_standalone_mode()
{
  struct fma_standalone_data *sdp;
  struct fma_myri *mp;
  struct fma_map_info *mip;

  if (A.debug) fma_log("Enter standalone mode");

  sdp = A.stand;
  mp = A.myri;
  mip = A.map_info;

  /* indicate we are in standalone mode */
  A.run_state = FMA_RUN_STANDALONE;
  fma_reset_flag_bits(FMA_FLAG_HAS_FMS);

  /* Active by default */
  sdp->active = TRUE;

  /* since we just lost fma contact, treat this as first mapping */
  mip->first_mapping = TRUE;

  /* set our level */
  fma_set_mapping_level(A.map_level);

  /* set us to be current mapper */
  fma_set_mapper(mp->my_max_mac, sdp->my_level);

  /* Start mapping, cleanest thing to do */
  fma_start_mapping_fabric(fma_standalone_map_done);
}

/*
 * This is called whenever we are entering a new run state from standalone
 */
void
fma_cancel_standalone_mode()
{
  struct fma_standalone_data *sdp;

  sdp = A.stand;

  /* Cancel any mapping that might be in progress */
  fma_mf_cancel_mapping();

  /* if we are waiting on a map, stop waiting now */
  if (sdp->map_wait_timer != NULL) {
    lf_remove_event(sdp->map_wait_timer);
    sdp->map_wait_timer = NULL;		/* indicate the timer has expired */
  }
}

/*
 * We got contact from another NIC out there.
 */
void
fma_standalone_got_contact(
  int level,
  lf_mac_addr_t max_mac_addr,
  struct fma_nic_info *nip,
  int port,
  uint8_t *route,
  int route_len,
  enum fma_nic_scout_type pkt_type)
{
  struct fma_standalone_data *sdp;
  struct fma_map_info *mip;
  int map_timeout;
  int cmp;

  sdp = A.stand;
  mip = A.map_info;

  /* If this mapper if higher than highest so far, then make this the
   * new mapper */
  cmp = fma_compare_levels(level, max_mac_addr,
      			   mip->mi_mapper_level, mip->mi_mapper_mac_addr);
  if (cmp > 0) {

    if (A.debug) {
      fma_log("Set new mapper from fma_standalone_got_contact");
    }

    /* set the current mapper */
    fma_set_mapper(max_mac_addr, level);

    /* If we used to be the active, cancel any mapping in progress and 
     * start waiting for a map */
    if (sdp->active) {
      sdp->active = FALSE;
      fma_mf_cancel_mapping();
    }

    /* remove any possible pending timeout events */
    if (sdp->map_wait_timer != NULL) {
      lf_remove_event(sdp->map_wait_timer);
      sdp->map_wait_timer = NULL;
    }

    /* Start waiting for a map from this new mapper */
    if (level > FMA_MAX_MAPPER_LEVEL) {
      map_timeout = FMA_FMS_MAP_WAIT_TIMEOUT;
    } else {
      map_timeout = FMA_STANDALONE_MAP_WAIT_TIMEOUT;
    }
    sdp->map_wait_timer = lf_schedule_event(fma_standalone_map_timeout, NULL,
		                            map_timeout);
    if (sdp->map_wait_timer == NULL) {
      LF_ERROR(("Error scheduling map wait timeout"));
    }

    /*
     * If this mapper has FMA contact, and we don't, use him as a proxy
     * to start communication with the FMS
     */
    if (level >= FMA_FMS_MAPPER_LEVEL) {
      fma_proxy_initiate(nip, port, route, route_len);
    }

  /* not higher, if this is a mapping NIC, cause a remap */
  } else if (cmp < 0 && pkt_type != FMA_NST_VERIFY) {
    lf_string_t why;

     /* If this is a packet from an fma that has already had a map, it means
      * his attempt to contact the previous mapper was unsuccessful, and we
      * are now back in the process of leader election.
      *
      * If pkt_type is FMA_NST_MAP, it means this packet is from an innocent
      * newly-started fma and a simple message to the active mapper should do
      * everything we need.
      */
    if (pkt_type == FMA_NST_REMAP && !sdp->active) {
      cmp = fma_compare_levels(level, max_mac_addr,
	  sdp->my_level, A.myri->my_max_mac);
      if (cmp < 0 && sdp->map_wait_timer == NULL) {
	fma_log("Scouted by lesser mapper - becoming active");
	sdp->active = TRUE;

	/* make me be the mapper */
	fma_set_mapper(A.myri->my_max_mac, sdp->my_level);

	if (sdp->map_wait_timer != NULL) {
	  lf_remove_event(sdp->map_wait_timer);
	  sdp->map_wait_timer = NULL;
	}
      }
    }

    /* If not mapping, then this map is invalid */
    if (!sdp->mapping_in_progress) {
      sprintf(why, "scouted by %s, lev=%d, pkt_type=%d",
	  fma_mac_to_hostname(max_mac_addr), level, pkt_type);
      fma_map_is_invalid(TRUE, why);
    }
  }
  return;

 except:
  fma_perror_exit(1);
}

/*
 * The map wait timeout expired - become the active mapper again and start
 * mapping!
 */
static void
fma_standalone_map_timeout(
  void *v)
{
  struct fma_standalone_data *sdp;

  sdp = A.stand;

  sdp->map_wait_timer = NULL;		/* indicate the timer has expired */

  fma_log("timeout waiting for map, going active again");

  /* go back to being the active mapper */
  sdp->active = TRUE;
  fma_set_mapper(A.myri->my_max_mac, sdp->my_level);

  fma_start_mapping_fabric(fma_standalone_map_done);
}

/*
 * Our existing map is invalid, we need a new one.
 * If we are the active mapper, start mapping, else send a request
 * to the active mapper to re-map
 */
void
fma_standalone_map_is_invalid(
  char *why)
{
  struct fma_standalone_data *sdp;
  struct fma_map_info *mip;
  int map_timeout;

  sdp = A.stand;
  mip = A.map_info;

  /* if we're the mapper, map it! */
  if (sdp->active) {
    fma_log("Mapping because: %s", why);
    fma_start_mapping_fabric(fma_standalone_map_done);

  /* otherwise, if we're not already waiting for a map, request one */
  } else if (sdp->map_wait_timer == NULL) {
    fma_request_remap_from_mapper(why);

    /* Start waiting for a map from this mapper */
    if (mip->mi_mapper_level > FMA_MAX_MAPPER_LEVEL) {
      map_timeout = FMA_FMS_MAP_WAIT_TIMEOUT;
    } else {
      map_timeout = FMA_STANDALONE_MAP_WAIT_TIMEOUT;
    }
    sdp->map_wait_timer = lf_schedule_event(fma_standalone_map_timeout, NULL,
		                            map_timeout);
    if (sdp->map_wait_timer == NULL) {
      LF_ERROR(("Error scheduling map wait timeout"));
    }
  }
  return;

 except:
  fma_perror_exit(1);
}

/*
 * Request a re-map from another standalone mapper
 */
static void
fma_request_remap_from_mapper(
  char *why)
{
  struct fma_standalone_data *sdp;
  struct fma_map_info *mip;
  struct fma_map_request req;
  struct lf_nic *map_nicp;
  struct lf_host *hp;

  sdp = A.stand;
  mip = A.map_info;

  /* don't issue a request if there is already one in progress */
  if (sdp->map_request_msg_id != 0) {
    if (A.debug) fma_log("remap request already in progress, skipping request");
  }

  /* fill in map request packet */
  req.hdr.type_16 = htons(FMA_PACKET_TYPE);
  req.hdr.subtype_16 = htons(FMA_SUBTYPE_MAP_FABRIC);
  req.map_version_32 = htonl(A.map_info->mi_map_version);
  strcpy(req.why, why);

  /* Make sure we know how to get to the mapper */
  map_nicp = lf_find_nic_by_mac(A.fabric, mip->mi_mapper_mac_addr);
  if (map_nicp != NULL) {
    hp = map_nicp->host;
  } else {
    hp = NULL;
  }
  if (hp != NULL && FMA_HOST(hp)->quick_route_len != -1) {

    fma_log("Requesting remap from %s: %s",
	fma_mac_to_hostname(mip->mi_mapper_mac_addr), why);

    sdp->map_request_msg_id = fma_tunnel_send(
	  &req, sizeof(req), NULL, 0,
	  FMA_HOST(hp)->quick_route, FMA_HOST(hp)->quick_route_len,
	  FMA_HOST(hp)->quick_nic, FMA_HOST(hp)->quick_port,
	  NULL, fma_standalone_map_request_timeout, NULL);
  } else {

    fma_log("Don't know how to reach mapper at %s",
	fma_mac_to_hostname(mip->mi_mapper_mac_addr));
  }
}

/*
 * Re-map request timed out - same as if map timed out
 */
static void
fma_standalone_map_request_timeout(
  void *v)
{
  struct fma_standalone_data *sdp;

  sdp = A.stand;

  if (A.debug) fma_log("Remap request timed out");
  sdp->map_request_msg_id = 0;

  if (sdp->map_wait_timer != NULL) {
    lf_remove_event(sdp->map_wait_timer);
    sdp->map_wait_timer = NULL;
  }

  /* Same as if remap itself timed out */
  fma_standalone_map_timeout(NULL);
}

/*
 * we got a map, clear the timer, make probe assignments
 */
void
fma_standalone_load_map(
  struct lf_topo_map *topo_map,
  int topo_map_size)
{
  struct fma_standalone_data *sdp;
  int rc;

  sdp = A.stand;

  /* If we are currently mapping, ignore this map */
  if (sdp->mapping_in_progress) {
    if (A.debug) {
      fma_log("Ignoring map from peer while mapping");
    }
    return;
  }

  /* first, clear all old verifies */
  fma_clear_verify_probes();

  /* Make a copy of this map and load it */
  rc = fma_copy_and_load_map(topo_map, topo_map_size);

  /* If map is valid, continue processing it */
  if (rc == 0) {

    /* re-distribute this map to other hosts */
    if (!sdp->active) {
      fma_distribute_topo_map(fma_report_dist_results);
    }

    /* Cancel remap request if in progress */
    if (sdp->map_request_msg_id != 0) {
      if (A.debug) fma_log("Cancelling remap request");
      fma_tunnel_cancel_send(sdp->map_request_msg_id);
      sdp->map_request_msg_id = 0;
    }

    /* Clear map receipt timeout timer */
    if (sdp->map_wait_timer != NULL) {
      lf_remove_event(sdp->map_wait_timer);
      sdp->map_wait_timer = NULL;
    }

    /* make new verify assignments */
    fma_assign_verifies();
  }
}

/*
 * Compare a given level/max_mac to the current mapper and return <0, 0, >0
 * a la memcmp()
 */
static int
fma_compare_levels(
  int level1,
  lf_mac_addr_t mac_addr1,
  int level2,
  lf_mac_addr_t mac_addr2)
{
  int cmp;

  cmp = level1 - level2;
  if (cmp != 0) return cmp;

  return LF_MAC_CMP(mac_addr1, mac_addr2);
}

/*
 * We have completed a map in standalone mode.  Start sending it to
 * everyone in it.
 */
void
fma_standalone_map_done(
  struct lf_topo_map *topo_map,
  int topo_map_size)
{
  struct fma_myri *fmp;
  struct fma_map_info *mip;
  struct fma_standalone_data *sdp;
  int rc;

  if (A.debug) fma_log("loading standalone map, size=%d bytes", topo_map_size);

  /* load pointers */
  fmp = A.myri;
  sdp = A.stand;
  mip = A.map_info;

  /* first, clear all old verify probes */
  fma_clear_verify_probes();

  /* load the map we generated */
  fma_load_map_struct(topo_map, topo_map_size);

  /* generate and load routes from this map */
  rc = fma_route_topo_map();
  if (rc != 0) {
    LF_ERROR(("Error routing from self-generated map"));
  }

  if (A.debug > 1) fma_dumpit();

  /* Send the topo map to everyone else who needs it */
  fma_distribute_topo_map(fma_local_map_distribution_complete);

  /* start a timer waiting for map distrobution */
  if (mip->mi_dist_global_remaining > 0) {
    sdp->map_dist_timer = lf_schedule_event(fma_standalone_dist_timeout, NULL,
	FMA_MAP_DISTRIBUTION_TIMEOUT);
    if (sdp->map_dist_timer == NULL) {
      LF_ERROR(("Error setting map distribution timeout"));
    }
  }

  /* Make verify probe assignments */
  fma_assign_verifies();

  return;

 except:
  fma_perror_exit(1);
}

/*
 * post-mapping grace period has expired
 */
static void
fma_mapping_grace_expired(
  void *v)
{
  A.stand->map_grace_timer = NULL;
  fma_reset_mapping_done();
}

/*
 * Called when time to reset mapping in progress
 */
void
fma_reset_mapping_done()
{
  struct fma_standalone_data *sdp;

  sdp = A.stand;

  /* clear grace timer if active */
  if (sdp->map_grace_timer != NULL) {
    lf_remove_event(sdp->map_grace_timer);
    sdp->map_grace_timer = NULL;
  }
  sdp->mapping_in_progress = FALSE;

  if (A.debug) fma_log("Resetting map_in_progress");
}

/*
 * Called when map distribution is complete
 */
static void
fma_local_map_distribution_complete()
{
  struct fma_map_info *mip;

  mip = A.map_info;

  /* subtract our distribution count */
  mip->mi_dist_global_remaining -= mip->mi_dist_total;

  if (A.debug) {
    fma_log("Local map distribution complete");
  }
  
  if (mip->mi_dist_global_remaining <= 0) {
    fma_standalone_map_distribution_complete();
  }
}

/*
 * Called when a remote node reports map distribution results
 */
void
fma_remote_map_dist_done(
  struct fma_dist_done_msg *msg,
  lf_mac_addr_t sender_mac)
{
  struct fma_map_info *mip;
  int done;
  int failed;

  mip = A.map_info;

  /* Make sure this is for correct map version */
  if (ntohl(msg->map_version_32) != mip->mi_map_version) {
    if (A.debug) {
      fma_log("Ignoring stale dist_done message from %s",
	  fma_mac_to_hostname(sender_mac));
    }
    return;
  }

  /* process the message */
  done = ntohl(msg->dist_cnt_32);
  failed = ntohl(msg->dist_failed_32);

  if (A.debug) {
    fma_log("%s reports map distribution done to %d nodes, %d failed",
	fma_mac_to_hostname(sender_mac), done, failed);
  }

  mip->mi_dist_global_remaining -= done;
  mip->mi_dist_failed += failed;

  /* If none left, call completion routine */
  if (mip->mi_dist_global_remaining <= 0) {
    fma_standalone_map_distribution_complete();
  }
}

/*
 * Called when all map distribution is complete
 */
static void
fma_standalone_map_distribution_complete()
{
  struct fma_standalone_data *sdp;
  struct fma_map_info *mip;

  sdp = A.stand;
  mip = A.map_info;

  fma_log("Local and global map distribution took %d seconds, %d failed",
      (int)(time(NULL) - mip->mi_dist_start_time), mip->mi_dist_failed);

  /* turn off timer if active */
  if (sdp->map_dist_timer != NULL) {
    lf_remove_event(sdp->map_dist_timer);
    sdp->map_dist_timer = NULL;		/* timer popped */
  }

  /* If we are the active mapper, start grace period */
  if (sdp->active) {
    sdp->map_grace_timer = lf_schedule_event(fma_mapping_grace_expired, NULL, 
	FMA_MAP_GRACE_PERIOD);
    if (sdp->map_grace_timer == NULL) {
      LF_ERROR(("Error starting mapping grace period timer"));
    }
  }
  return;

 except:
  fma_perror_exit(1);
}

static void
fma_standalone_dist_timeout(
  void *v)
{
  struct fma_standalone_data *sdp;
  struct fma_map_info *mip;

  sdp = A.stand;
  mip = A.map_info;

  sdp->map_dist_timer = NULL;		/* timer popped */

  fma_log("Global map distribution timed out with %d remaining, proceeding",
      mip->mi_dist_global_remaining);

  /* treat any remaining as failures */
  mip->mi_dist_failed += mip->mi_dist_global_remaining;
  mip->mi_dist_global_remaining = 0;

  /* call completion routine */
  fma_standalone_map_distribution_complete();
}

/*
 * Tell the active mapper how many maps we distributed
 */
static void
fma_report_dist_results()
{
  struct fma_dist_done_msg msg;
  struct fma_map_info *mip;
  struct lf_nic *map_nicp;
  struct lf_host *hp;

  mip = A.map_info;

  /* Only bother with this if we actually did something */
  if (mip->mi_dist_total <= 0) {
    return;
  }

  msg.hdr.type_16 = htons(FMA_PACKET_TYPE);
  msg.hdr.subtype_16 = htons(FMA_SUBTYPE_DIST_DONE);

  msg.map_version_32 = htonl(mip->mi_map_version);
  msg.dist_cnt_32 = htonl(mip->mi_dist_total);
  msg.dist_failed_32 = htonl(mip->mi_dist_failed);

  /* Send report back to mapper */

  /* Make sure we know how to get to the mapper */
  map_nicp = lf_find_nic_by_mac(A.fabric, mip->mi_mapper_mac_addr);
  if (map_nicp != NULL) {
    hp = map_nicp->host;
  } else {
    hp = NULL;
  }
  if (hp != NULL && FMA_HOST(hp)->quick_route_len != -1) {

    fma_log("Reporting distribution results: %d done, %d failed",
	mip->mi_dist_total, mip->mi_dist_failed);

    (void) fma_tunnel_send(&msg, sizeof(msg), NULL, 0,
	  FMA_HOST(hp)->quick_route, FMA_HOST(hp)->quick_route_len,
	  FMA_HOST(hp)->quick_nic, FMA_HOST(hp)->quick_port,
	  NULL, NULL, NULL);
  } else {

    fma_log("Don't know how to reach mapper at %s",
	fma_mac_to_hostname(mip->mi_mapper_mac_addr));
  }
}

#if 0
/*
 * Test whether a given topo link is xbar-xbar or nor
 */
static int
fma_is_link_xbar_xbar(
  int topo_index)
{
  struct lf_fabric *fp;
  struct lf_xbar *xp1;
  struct lf_xbar *xp2;
  int p1;

  fp = A.fabric;
  
  /* Get pointer to first end of the link */
  xp1 = LF_XBAR(FMA_FABRIC(fp)->link_end_node[topo_index]);
  p1 = FMA_FABRIC(fp)->link_end_port[topo_index];

  /* Make this end is an xbar */
  if (xp1->ln_type != LF_NODE_XBAR) {
    return FALSE;
  }

  /* get the other end */
  xp2 = LF_XBAR(xp1->topo_ports[p1]);

  /* Make sure it is an xbar */
  if (xp2->ln_type != LF_NODE_XBAR) {
    return FALSE;
  }

  return TRUE;
}
#endif
